// Professional Navbar System for TheL4FPRO
class ProfessionalNavbar {
    constructor() {
        this.isMenuOpen = false;
        this.isUserMenuOpen = false;
        this.currentUser = null;
        
        this.init();
    }

    init() {
        this.getCurrentUser();
        this.setupScrollEffects();
        this.setupMobileMenu();
        this.setupUserMenu();
        this.setupActiveStates();
        this.setupScrollProgress();
        this.setupFloatingCTA();
    }

    getCurrentUser() {
        // Sync with existing auth system
        this.currentUser = JSON.parse(localStorage.getItem('currentUser') || 'null');
        this.updateNavigation();
    }

    updateNavigation() {
        const navbar = document.querySelector('.navbar');
        if (!navbar) return;

        const navMenu = navbar.querySelector('.nav-menu');
        const userMenuWrapper = navbar.querySelector('.user-menu-wrapper');
        
        if (!navMenu) return;

        // Clear existing dynamic content
        navMenu.querySelectorAll('.nav-dynamic').forEach(item => item.remove());
        
        if (this.currentUser) {
            this.renderAuthenticatedNav(navMenu, userMenuWrapper);
        } else {
            this.renderGuestNav(navMenu);
        }
        
        this.setupActiveStates();
    }

    renderGuestNav(navMenu) {
        // For non-authenticated users: Home, Login, Register (Academy)
        const guestItems = [
            { href: '/', icon: 'fas fa-home', text: 'Inicio', class: '' },
            { href: '/login', icon: 'fas fa-sign-in-alt', text: 'Iniciar Sesión', class: 'btn-outline' },
            { href: '/academy-register', icon: 'fas fa-graduation-cap', text: 'Únete a la Academia', class: 'btn-style' }
        ];

        // Remove authenticated-only items
        navMenu.querySelectorAll('.nav-auth-only').forEach(item => item.remove());

        // Add guest items
        guestItems.forEach(item => {
            const li = document.createElement('li');
            li.className = 'nav-dynamic';
            li.innerHTML = `
                <a href="${item.href}" class="nav-link ${item.class}">
                    <i class="${item.icon}"></i>
                    <span>${item.text}</span>
                </a>
            `;
            navMenu.appendChild(li);
        });

        // Hide user menu wrapper
        const userMenuWrapper = document.querySelector('.user-menu-wrapper');
        if (userMenuWrapper) {
            userMenuWrapper.style.display = 'none';
        }
    }

    renderAuthenticatedNav(navMenu, userMenuWrapper) {
        // For authenticated users: Full navigation based on role
        const baseItems = [
            { href: '/', icon: 'fas fa-home', text: 'Inicio' },
            { href: '/ticketing', icon: 'fas fa-calendar-alt', text: 'Reservas' }
        ];

        // Role-specific items
        if (this.currentUser.role === 'docente') {
            baseItems.push({ href: '/docente', icon: 'fas fa-chalkboard-teacher', text: 'Panel Docente' });
        }

        if (this.currentUser.role === 'admin') {
            baseItems.push({ href: '/admin', icon: 'fas fa-shield-alt', text: 'Administración' });
        }

        // Clear and rebuild nav items
        navMenu.innerHTML = '';
        
        baseItems.forEach(item => {
            const li = document.createElement('li');
            li.className = 'nav-dynamic nav-auth-only';
            li.innerHTML = `
                <a href="${item.href}" class="nav-link">
                    <i class="${item.icon}"></i>
                    <span>${item.text}</span>
                </a>
            `;
            navMenu.appendChild(li);
        });

        // Setup user menu
        this.setupUserMenuContent(userMenuWrapper);
    }

    setupUserMenuContent(userMenuWrapper) {
        if (!userMenuWrapper || !this.currentUser) return;

        userMenuWrapper.style.display = 'block';
        
        const userName = this.currentUser.name || 'Usuario';
        const userEmail = this.currentUser.email || '';
        const userRole = this.currentUser.role || 'alumno';
        const userInitial = userName.charAt(0).toUpperCase();

        userMenuWrapper.innerHTML = `
            <div class="user-toggle" id="userToggle">
                <div class="user-avatar">${userInitial}</div>
                <div class="user-info">
                    <div class="user-name">${userName}</div>
                    <div class="user-role">${userRole}</div>
                </div>
                <i class="fas fa-chevron-down dropdown-icon"></i>
            </div>
            <div class="user-dropdown" id="userDropdown">
                <div class="dropdown-header">
                    <div class="dropdown-avatar">${userInitial}</div>
                    <div class="dropdown-info">
                        <div class="dropdown-name">${userName}</div>
                        <div class="dropdown-email">${userEmail}</div>
                    </div>
                </div>
                <div class="dropdown-items">
                    <a href="/perfil" class="dropdown-item">
                        <i class="fas fa-user-circle"></i>
                        <span>Mi Perfil</span>
                    </a>
                    <a href="/ajustes" class="dropdown-item">
                        <i class="fas fa-cog"></i>
                        <span>Configuración</span>
                    </a>
                    <a href="/notificaciones" class="dropdown-item">
                        <i class="fas fa-bell"></i>
                        <span>Notificaciones</span>
                    </a>
                    <a href="/notes" class="dropdown-item">
                        <i class="fas fa-sticky-note"></i>
                        <span>Mis Notas</span>
                    </a>
                    <a href="/logout" class="dropdown-item logout">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Cerrar Sesión</span>
                    </a>
                </div>
            </div>
        `;

        // Setup user menu toggle
        const userToggle = userMenuWrapper.querySelector('#userToggle');
        const userDropdown = userMenuWrapper.querySelector('#userDropdown');

        if (userToggle && userDropdown) {
            userToggle.addEventListener('click', (e) => {
                e.stopPropagation();
                this.toggleUserMenu();
            });

            // Close dropdown when clicking outside
            document.addEventListener('click', (e) => {
                if (!userMenuWrapper.contains(e.target)) {
                    this.closeUserMenu();
                }
            });
        }
    }

    setupScrollEffects() {
        let ticking = false;
        
        const updateNavbar = () => {
            const navbar = document.querySelector('.navbar');
            if (!navbar) return;
            
            if (window.scrollY > 50) {
                navbar.classList.add('scrolled');
            } else {
                navbar.classList.remove('scrolled');
            }
            
            ticking = false;
        };

        window.addEventListener('scroll', () => {
            if (!ticking) {
                requestAnimationFrame(updateNavbar);
                ticking = true;
            }
        });
    }

    setupMobileMenu() {
        const hamburger = document.getElementById('hamburger');
        const navMenu = document.getElementById('navMenu');

        if (hamburger && navMenu) {
            hamburger.addEventListener('click', (e) => {
                e.stopPropagation();
                this.toggleMobileMenu();
            });

            // Close menu when clicking on a link
            navMenu.querySelectorAll('.nav-link').forEach(link => {
                link.addEventListener('click', () => {
                    this.closeMobileMenu();
                });
            });

            // Close menu when clicking outside
            document.addEventListener('click', (e) => {
                if (!hamburger.contains(e.target) && !navMenu.contains(e.target)) {
                    this.closeMobileMenu();
                }
            });
        }
    }

    setupUserMenu() {
        // User menu setup is handled in setupUserMenuContent
        // This method can be used for additional user menu functionality
    }

    setupActiveStates() {
        const currentPath = window.location.pathname;
        const navLinks = document.querySelectorAll('.nav-link');
        
        navLinks.forEach(link => {
            const href = link.getAttribute('href');
            
            // Remove active class from all links
            link.classList.remove('active');
            
            // Add active class to current page
            if (href === currentPath || 
                (currentPath === '/' && href === '/') ||
                (currentPath.includes(href) && href !== '/')) {
                link.classList.add('active');
            }
        });
    }

    setupScrollProgress() {
        // Create scroll progress indicator
        if (document.querySelector('.scroll-progress')) return;
        
        const progressBar = document.createElement('div');
        progressBar.className = 'scroll-progress';
        progressBar.innerHTML = '<div class="progress-fill"></div>';
        document.body.appendChild(progressBar);
        
        const progressFill = progressBar.querySelector('.progress-fill');
        
        let ticking = false;
        const updateProgress = () => {
            const scrollHeight = document.documentElement.scrollHeight - window.innerHeight;
            const scrolled = Math.min(100, (window.pageYOffset / scrollHeight) * 100);
            progressFill.style.width = `${scrolled}%`;
            ticking = false;
        };
        
        window.addEventListener('scroll', () => {
            if (!ticking) {
                requestAnimationFrame(updateProgress);
                ticking = true;
            }
        });
    }

    setupFloatingCTA() {
        // Create floating CTA for guest users
        if (this.currentUser) return;
        
        if (document.querySelector('.floating-nav-cta')) return;
        
        const floatingCTA = document.createElement('a');
        floatingCTA.href = '/academy-register';
        floatingCTA.className = 'floating-nav-cta';
        floatingCTA.innerHTML = '<i class="fas fa-graduation-cap"></i>';
        floatingCTA.title = 'Únete a la Academia';
        
        document.body.appendChild(floatingCTA);
        
        // Show on scroll
        let ticking = false;
        const updateCTA = () => {
            if (window.scrollY > 300) {
                floatingCTA.classList.add('show');
            } else {
                floatingCTA.classList.remove('show');
            }
            ticking = false;
        };
        
        window.addEventListener('scroll', () => {
            if (!ticking) {
                requestAnimationFrame(updateCTA);
                ticking = true;
            }
        });
    }

    toggleMobileMenu() {
        const hamburger = document.getElementById('hamburger');
        const navMenu = document.getElementById('navMenu');
        
        this.isMenuOpen = !this.isMenuOpen;
        
        if (hamburger) hamburger.classList.toggle('active', this.isMenuOpen);
        if (navMenu) navMenu.classList.toggle('active', this.isMenuOpen);
        
        // Prevent body scroll when menu is open
        document.body.style.overflow = this.isMenuOpen ? 'hidden' : '';
    }

    closeMobileMenu() {
        const hamburger = document.getElementById('hamburger');
        const navMenu = document.getElementById('navMenu');
        
        this.isMenuOpen = false;
        
        if (hamburger) hamburger.classList.remove('active');
        if (navMenu) navMenu.classList.remove('active');
        
        document.body.style.overflow = '';
    }

    toggleUserMenu() {
        const userToggle = document.querySelector('#userToggle');
        const userDropdown = document.querySelector('#userDropdown');
        
        this.isUserMenuOpen = !this.isUserMenuOpen;
        
        if (userToggle) userToggle.classList.toggle('active', this.isUserMenuOpen);
        if (userDropdown) userDropdown.classList.toggle('active', this.isUserMenuOpen);
    }

    closeUserMenu() {
        const userToggle = document.querySelector('#userToggle');
        const userDropdown = document.querySelector('#userDropdown');
        
        this.isUserMenuOpen = false;
        
        if (userToggle) userToggle.classList.remove('active');
        if (userDropdown) userDropdown.classList.remove('active');
    }

    // Method to refresh navbar when user state changes
    refresh() {
        this.getCurrentUser();
    }

    // Smooth navigation with page transitions
    navigateToPage(url) {
        if (window.campoVerdeApp && window.campoVerdeApp.navigateToPage) {
            window.campoVerdeApp.navigateToPage(url);
        } else {
            window.location.href = url;
        }
    }
}

// Initialize navbar when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.professionalNavbar = new ProfessionalNavbar();
    
    // Listen for auth state changes
    window.addEventListener('storage', (e) => {
        if (e.key === 'currentUser') {
            window.professionalNavbar.refresh();
        }
    });
    
    // Integrate with existing auth system
    if (window.campoVerdeApp) {
        const originalUpdateAuthStatus = window.campoVerdeApp.updateAuthStatus;
        window.campoVerdeApp.updateAuthStatus = function() {
            if (originalUpdateAuthStatus) {
                originalUpdateAuthStatus.call(this);
            }
            if (window.professionalNavbar) {
                window.professionalNavbar.refresh();
            }
        };
    }
});

// Export for use by other scripts
window.ProfessionalNavbar = ProfessionalNavbar;