const qs = (s, r=document) => r.querySelector(s);
const qsa = (s, r=document) => Array.from(r.querySelectorAll(s));
const fmtEur = n => (n/100).toLocaleString('es-ES', {style:'currency', currency:'EUR'});
const parseEurToCents = (txt) => Math.round(parseFloat(String(txt || 0).replace(',', '.')) * 100) || 0;

function getCSRF() {
  const m = document.querySelector('meta[name="csrf-token"]');
  return m ? m.getAttribute('content') : '';
}

async function apiFetch(url, opts={}) {
  const o = Object.assign({headers:{}}, opts);
  o.headers['Accept'] = 'application/json';
  if (o.body && !(o.body instanceof FormData)) {
    o.headers['Content-Type'] = 'application/json';
  }
  if (['POST','PATCH','PUT','DELETE'].includes((o.method||'GET').toUpperCase())) {
    const csrf = getCSRF();
    if (csrf) o.headers['X-CSRF-Token'] = csrf;
  }
  const res = await fetch(url, o);
  if (!res.ok) {
    let detail = '';
    try { const j = await res.json(); detail = j && j.detail ? `: ${j.detail}` : ''; } catch {}
    throw new Error(`HTTP ${res.status}${detail}`);
  }
  const ct = res.headers.get('Content-Type') || '';
  if (ct.includes('application/json')) return res.json();
  return res.text();
}

function monthRangeYYYYMM(monthStr) {
  const [yyyy, mm] = monthStr.split('-').map(x => parseInt(x, 10));
  const start = new Date(yyyy, mm - 1, 1);
  const end = new Date(yyyy, mm, 0);
  const pad = (n)=> String(n).padStart(2,'0');
  return {
    from: `${start.getFullYear()}-${pad(start.getMonth()+1)}-01`,
    to: `${end.getFullYear()}-${pad(end.getMonth()+1)}-${pad(end.getDate())}`
  };
}

function amountCentsFromReservation(r) {
  if (r && r.price_cents != null && !isNaN(r.price_cents)) {
    return parseInt(r.price_cents, 10) || 0;
  }
  if (r && r.amount_cents != null && !isNaN(r.amount_cents)) {
    return parseInt(r.amount_cents, 10) || 0;
  }
  const euroLike = r?.price_eur ?? r?.price ?? r?.amount ?? r?.importe_eur;
  if (euroLike != null && euroLike !== '') {
    const v = parseFloat(String(euroLike).replace(',', '.'));
    if (!isNaN(v)) return Math.round(v * 100);
  }
  return 0;
}

function renderIncomeDetails(rows, teacherById) {
  const tbody = qs('#incomeDetailsTable tbody');
  if (!tbody) return;
  tbody.innerHTML = '';
  const sorted = [...(rows || [])].sort((a, b) => {
    const da = `${a.date || ''} ${a.time || ''}`;
    const db = `${b.date || ''} ${b.time || ''}`;
    return da.localeCompare(db);
  });
  sorted.forEach(r => {
    const tr = document.createElement('tr');
    const alumno = (r.name && String(r.name).trim()) || (r.email && String(r.email).trim()) || '—';
    const docente = (r.docente_id != null && teacherById && teacherById[r.docente_id] && teacherById[r.docente_id].name) ? teacherById[r.docente_id].name : '—';
    const importeCents = amountCentsFromReservation(r);
    tr.innerHTML = `
      <td>${r.date || '—'}</td>
      <td>${r.time || '—'}</td>
      <td>${alumno}</td>
      <td>${docente}</td>
      <td>${fmtEur(importeCents)}</td>
    `;
    tbody.appendChild(tr);
  });
}

async function loadIncomes(monthStr) {
  const {from, to} = monthRangeYYYYMM(monthStr);
  const [reservations, teachers] = await Promise.all([
    apiFetch(`/api/reservations?status=confirmada&date_from=${encodeURIComponent(from)}&date_to=${encodeURIComponent(to)}`),
    apiFetch(`/api/teachers`)
  ]);

  const teacherById = {};
  (teachers || []).forEach(t => { teacherById[t.id] = t; });

  const rows = Array.isArray(reservations) ? reservations : [];
  const totalBookings = rows.length;
  const totalRevenueCents = rows.reduce((acc, r) => acc + amountCentsFromReservation(r), 0);
  const avgCents = totalBookings ? Math.round(totalRevenueCents / totalBookings) : 0;

  const tb = qs('#totalBookings'); if (tb) tb.textContent = String(totalBookings);
  const trv = qs('#totalRevenue'); if (trv) trv.textContent = fmtEur(totalRevenueCents);
  const tia = qs('#docenteIncomeAvg'); if (tia) tia.textContent = fmtEur(avgCents);

  const agg = new Map();
  rows.forEach(r => {
    const did = r.docente_id || 0;
    const cur = agg.get(did) || { docente_id: did, name: (teacherById[did]?.name || '—'), count: 0, sum: 0 };
    cur.count += 1;
    cur.sum += amountCentsFromReservation(r);
    agg.set(did, cur);
  });

  const tbody = qs('#docenteIncomeTable tbody');
  if (tbody) {
    tbody.innerHTML = '';
    const sorted = Array.from(agg.values()).sort((a,b) => b.sum - a.sum);
    sorted.forEach(r => {
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>${r.name}</td>
        <td>${r.count}</td>
        <td>${fmtEur(r.sum)}</td>
      `;
      tbody.appendChild(tr);
    });
    const totalCount = Array.from(agg.values()).reduce((a,r)=>a+r.count,0);
    const totalSum = Array.from(agg.values()).reduce((a,r)=>a+r.sum,0);
    const dib = qs('#docenteIncomeBookings'); if (dib) dib.textContent = String(totalCount);
    const dit = qs('#docenteIncomeTotal'); if (dit) dit.textContent = fmtEur(totalSum);
  }

  renderIncomeDetails(rows, teacherById);

  return {rows, teachers: teacherById, totalRevenueCents, totalBookings};
}

function filterExpensesRows(term) {
  const t = (term||'').trim().toLowerCase();
  qsa('#expensesTable tbody tr').forEach(tr => {
    const txt = tr.textContent.toLowerCase();
    tr.style.display = txt.includes(t) ? '' : 'none';
  });
}

function expenseRowHTML(e) {
  const d = (e.date || '').slice(0,10);
  const alumno = e.user_name || e.username || e.user || '—';
  const concept = e.concept || '';
  const category = e.category || '';
  const amt = fmtEur(parseInt(e.amount_cents||0,10) || 0);
  return `
    <tr data-id="${e.id}">
      <td>${d}</td>
      <td>${alumno}</td>
      <td>${concept}</td>
      <td>${category}</td>
      <td>${amt}</td>
      <td class="align-right">
        <button class="btn btn-secondary js-del-expense" title="Eliminar"><i class="fa-regular fa-trash-can"></i></button>
      </td>
    </tr>
  `;
}

function updateExpensesSummary() {
  const rows = qsa('#expensesTable tbody tr').filter(tr => tr.style.display !== 'none');
  const vals = rows.map(tr => {
    const text = tr.children[4].textContent || '0';
    return parseEurToCents(text.replace(/[^\d,.-]/g,''));
  });
  const count = rows.length;
  const sum = vals.reduce((a,b)=>a+b,0);
  const avg = count ? Math.round(sum / count) : 0;
  const et = qs('#expensesTotal'); if (et) et.textContent = fmtEur(sum);
  const ec = qs('#expensesCount'); if (ec) ec.textContent = String(count);
  const ea = qs('#expensesAvg'); if (ea) ea.textContent = fmtEur(avg);
}

async function loadExpenses(monthStr) {
  const {from, to} = monthRangeYYYYMM(monthStr);
  const list = await apiFetch(`/api/expenses?date_from=${encodeURIComponent(from)}&date_to=${encodeURIComponent(to)}`);
  const tbody = qs('#expensesTable tbody');
  if (!tbody) return;
  tbody.innerHTML = '';
  (list || []).sort((a,b)=> (a.date||'').localeCompare(b.date||'')).forEach(e => {
    tbody.insertAdjacentHTML('beforeend', expenseRowHTML(e));
  });
  updateExpensesSummary();
}

async function createExpense(payload) {
  await apiFetch('/api/expenses', {
    method: 'POST',
    body: JSON.stringify(payload)
  });
}

async function deleteExpense(id) {
  await apiFetch(`/api/expenses/${id}`, { method: 'DELETE' });
}

function openExpenseModal() {
  const m = qs('#userExpensesModal');
  if (!m) return;
  m.classList.add('active');
  const form = qs('#expenseForm');
  if (!form) return;
  form.reset();
  const today = new Date();
  const pad = n => String(n).padStart(2,'0');
  if (form.elements.date) form.elements.date.value = `${today.getFullYear()}-${pad(today.getMonth()+1)}-${pad(today.getDate())}`;
}

function closeExpenseModal() {
  const m = qs('#userExpensesModal');
  if (m) m.classList.remove('active');
}

function serializeExpenseForm(form) {
  return {
    date: form.elements.date.value,
    amount_cents: parseEurToCents(form.elements.amount_eur.value),
    category: (form.elements.category.value || '').trim(),
    concept: (form.elements.concept.value || '').trim(),
    notes: (form.elements.notes.value || '').trim()
  };
}

function exportTableToCSV(rows, headers, filename) {
  const esc = (v)=> `"${String(v).replaceAll('"','""')}"`;
  const csv = [headers.map(esc).join(',')].concat(
    rows.map(r => headers.map(h => esc(r[h])).join(','))
  ).join('\n');
  const blob = new Blob([csv], {type:'text/csv;charset=utf-8;'});
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = filename;
  a.click();
  URL.revokeObjectURL(url);
}

async function exportDocenteIncomeCSV(monthStr) {
  const {rows, teachers} = await loadIncomes(monthStr);
  const agg = {};
  rows.forEach(r => {
    const did = r.docente_id || 0;
    const key = String(did);
    agg[key] = agg[key] || {docente: (teachers[did]?.name || '—'), reservas: 0, importe_eur: 0};
    agg[key].reservas += 1;
    agg[key].importe_eur += amountCentsFromReservation(r)/100;
  });
  const out = Object.values(agg).map(x => ({docente: x.docente, reservas: x.reservas, importe: x.importe_eur.toFixed(2)}));
  exportTableToCSV(out, ['docente','reservas','importe'], `ingresos_docentes_${monthStr}.csv`);
}

function exportExpensesCSV() {
  const rows = qsa('#expensesTable tbody tr').filter(tr => tr.style.display !== 'none').map(tr => {
    return {
      fecha: tr.children[0].textContent.trim(),
      alumno: tr.children[1].textContent.trim(),
      concepto: tr.children[2].textContent.trim(),
      categoria: tr.children[3].textContent.trim(),
      importe: tr.children[4].textContent.trim().replace(/\s/g,'')
    };
  });
  exportTableToCSV(rows, ['fecha','alumno','concepto','categoria','importe'], `gastos_${qs('#incomeMonth').value}.csv`);
}

async function exportAllCSV() {
  const monthInput = qs('#incomeMonth');
  const monthStr = monthInput ? monthInput.value : '';
  const { rows: incomes, teachers } = await loadIncomes(monthStr);
  const { from, to } = monthRangeYYYYMM(monthStr);
  const expenses = await apiFetch(`/api/expenses?date_from=${encodeURIComponent(from)}&date_to=${encodeURIComponent(to)}`);
  const esc = (v)=> `"${String(v).replaceAll('"','""')}"`;
  let csv = "INGRESOS\n";
  csv += ["Fecha","Hora","Alumno","Docente","Importe (€)"].map(esc).join(',') + "\n";
  incomes.forEach(r => {
    const alumno = (r.name || r.email || "—");
    const docente = (teachers && teachers[r.docente_id]?.name) || "—";
    const importe = (amountCentsFromReservation(r) / 100).toFixed(2);
    csv += [r.date, r.time, alumno, docente, importe].map(esc).join(',') + "\n";
  });
  csv += "\n\nGASTOS\n";
  csv += ["Fecha","Alumno","Concepto","Categoría","Importe (€)"].map(esc).join(',') + "\n";
  (expenses || []).forEach(e => {
    const importe = (e.amount_cents / 100).toFixed(2);
    csv += [e.date, e.user_name || e.username || e.user || "—", e.concept, e.category, importe].map(esc).join(',') + "\n";
  });
  const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = `contabilidad_${monthStr}.csv`;
  a.click();
  URL.revokeObjectURL(url);
}

async function boot() {
  const now = new Date();
  const pad = n => String(n).padStart(2,'0');
  const yyyyMM = `${now.getFullYear()}-${pad(now.getMonth()+1)}`;
  const monthInput = qs('#incomeMonth');
  if (monthInput) monthInput.value = yyyyMM;

  await loadIncomes(monthInput ? monthInput.value : yyyyMM);
  await loadExpenses(monthInput ? monthInput.value : yyyyMM);

  if (monthInput) {
    monthInput.addEventListener('change', async () => {
      await loadIncomes(monthInput.value);
      await loadExpenses(monthInput.value);
    });
  }

  const btnExportAll = qs('#exportAll');
  if (btnExportAll) btnExportAll.addEventListener('click', exportAllCSV);

  const btnRefreshIncomes = qs('#refreshDocenteIncome');
  if (btnRefreshIncomes) btnRefreshIncomes.addEventListener('click', async () => {
    await loadIncomes(monthInput.value);
  });

  const btnRefreshExpenses = qs('#refreshExpenses');
  if (btnRefreshExpenses) btnRefreshExpenses.addEventListener('click', async () => {
    await loadExpenses(monthInput.value);
  });

  const btnExportDocenteIncome = qs('#exportDocenteIncome');
  if (btnExportDocenteIncome) btnExportDocenteIncome.addEventListener('click', async () => {
    await exportDocenteIncomeCSV(monthInput.value);
  });

  const btnExportExpenses = qs('#exportExpenses');
  if (btnExportExpenses) btnExportExpenses.addEventListener('click', () => {
    exportExpensesCSV();
  });

  const btnOpenExpense = qs('#openNewExpense');
  if (btnOpenExpense) btnOpenExpense.addEventListener('click', openExpenseModal);

  const btnCloseExpensesModal = qs('#closeExpensesModal');
  if (btnCloseExpensesModal) btnCloseExpensesModal.addEventListener('click', closeExpenseModal);

  const btnCloseExpensesModalBtn = qs('#closeExpensesModalBtn');
  if (btnCloseExpensesModalBtn) btnCloseExpensesModalBtn.addEventListener('click', closeExpenseModal);

  const expenseForm = qs('#expenseForm');
  if (expenseForm) {
    expenseForm.addEventListener('submit', async (e) => {
      e.preventDefault();
      const payload = serializeExpenseForm(e.currentTarget);
      await createExpense(payload);
      closeExpenseModal();
      await loadExpenses(monthInput.value);
    });
  }

  const searchExpenses = qs('#searchExpenses');
  if (searchExpenses) {
    searchExpenses.addEventListener('input', () => {
      filterExpensesRows(searchExpenses.value);
      updateExpensesSummary();
    });
  }

  const expensesTable = qs('#expensesTable');
  if (expensesTable) {
    expensesTable.addEventListener('click', async (e) => {
      const btn = e.target.closest('.js-del-expense');
      if (!btn) return;
      const tr = e.target.closest('tr');
      const id = tr?.getAttribute('data-id');
      if (!id) return;
      await deleteExpense(id);
      tr.remove();
      updateExpensesSummary();
    });
  }
}

document.addEventListener('DOMContentLoaded', boot);
