// Authentication JavaScript for TheL4FPRO
class AuthSystem {
    constructor() {
        this.users = JSON.parse(localStorage.getItem('users') || '[]');
        this.currentUser = JSON.parse(localStorage.getItem('currentUser') || 'null'); 
        
        this.init();
    }

    init() {
        this.setupAuthSwitcher();
        this.setupForms();
        this.setupPasswordToggles();
        this.setupSocialAuth();
        this.setupFormValidation();
    }

    setupAuthSwitcher() {
        const switchButtons = document.querySelectorAll('.switch-btn');
        const indicator = document.querySelector('.switch-indicator');
        const loginCard = document.getElementById('loginCard');
        const registerCard = document.getElementById('registerCard');

        switchButtons.forEach(btn => {
            btn.addEventListener('click', () => {
                const formType = btn.getAttribute('data-form');
                
                // Update active button
                switchButtons.forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
                
                // Move indicator
                const btnRect = btn.getBoundingClientRect();
                const containerRect = btn.parentElement.getBoundingClientRect();
                const leftPosition = btnRect.left - containerRect.left;
                indicator.style.transform = `translateX(${leftPosition}px)`;
                indicator.style.width = `${btnRect.width}px`;
                
                // Switch forms
                if (formType === 'login') {
                    loginCard.classList.add('active');
                    registerCard.classList.remove('active');
                } else {
                    registerCard.classList.add('active');
                    loginCard.classList.remove('active');
                }
            });
        });

        // Initialize indicator position
        const activeBtn = document.querySelector('.switch-btn.active');
        if (activeBtn) {
            setTimeout(() => {
                const btnRect = activeBtn.getBoundingClientRect();
                const containerRect = activeBtn.parentElement.getBoundingClientRect();
                const leftPosition = btnRect.left - containerRect.left;
                indicator.style.transform = `translateX(${leftPosition}px)`;
                indicator.style.width = `${btnRect.width}px`;
            }, 100);
        }
    }

    setupForms() {
        const loginForm = document.getElementById('loginForm');
        const registerForm = document.getElementById('registerForm');

        if (loginForm) {
            loginForm.addEventListener('submit', (e) => this.handleLogin(e));
        }

        if (registerForm) {
            registerForm.addEventListener('submit', (e) => this.handleRegister(e));
        }
    }

    setupPasswordToggles() {
        const toggleButtons = document.querySelectorAll('.password-toggle');
        
        toggleButtons.forEach(btn => {
            btn.addEventListener('click', () => {
                const targetId = btn.getAttribute('data-target');
                const targetInput = document.getElementById(targetId);
                const icon = btn.querySelector('i');
                
                if (targetInput.type === 'password') {
                    targetInput.type = 'text';
                    icon.className = 'fas fa-eye-slash';
                } else {
                    targetInput.type = 'password';
                    icon.className = 'fas fa-eye';
                }
            });
        });
    }

    setupSocialAuth() {
        const socialButtons = document.querySelectorAll('.social-btn');
        
        socialButtons.forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const provider = btn.classList.contains('google') ? 'Google' : 'Facebook';
                this.handleSocialAuth(provider);
            });
        });
    }

    setupFormValidation() {
        const inputs = document.querySelectorAll('input');
        
        inputs.forEach(input => {
            // Real-time validation
            input.addEventListener('blur', () => this.validateField(input));
            input.addEventListener('input', () => this.clearError(input));
            
            // Enhanced input animations
            input.addEventListener('focus', () => {
                input.parentElement.classList.add('focused');
            });
            
            input.addEventListener('blur', () => {
                if (!input.value) {
                    input.parentElement.classList.remove('focused');
                }
            });
            
            // Check if input has value on load
            if (input.value) {
                input.parentElement.classList.add('focused');
            }
        });
    }

    validateField(field) {
        const value = field.value.trim();
        const errorElement = field.closest('.form-group').querySelector('.form-error');
        let isValid = true;
        let errorMessage = '';

        // Required field check
        if (field.required && !value) {
            isValid = false;
            errorMessage = 'Este campo es obligatorio';
        } else {
            // Type-specific validation
            switch(field.type) {
                case 'email':
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (value && !emailRegex.test(value)) {
                        isValid = false;
                        errorMessage = 'Por favor, introduce un email válido';
                    }
                    break;
                
                case 'password':
                    if (value && value.length < 6) {
                        isValid = false;
                        errorMessage = 'La contraseña debe tener al menos 6 caracteres';
                    }
                    break;
                
                case 'tel':
                    const phoneRegex = /^\+?[\d\s\-\(\)]{9,}$/;
                    if (value && !phoneRegex.test(value)) {
                        isValid = false;
                        errorMessage = 'Por favor, introduce un teléfono válido';
                    }
                    break;
                
                case 'text':
                    if (field.required && value && value.length < 2) {
                        isValid = false;
                        errorMessage = 'Este campo debe tener al menos 2 caracteres';
                    }
                    break;
            }
        }

        // Password confirmation
        if (field.id === 'registerConfirmPassword') {
            const password = document.getElementById('registerPassword').value;
            if (value && value !== password) {
                isValid = false;
                errorMessage = 'Las contraseñas no coinciden';
            }
        }

        // Email uniqueness check for registration
        if (field.id === 'registerEmail' && value) {
            const existingUser = this.users.find(u => u.email === value);
            if (existingUser) {
                isValid = false;
                errorMessage = 'Este email ya está registrado';
            }
        }

        this.setFieldState(field, isValid, errorMessage);
        return isValid;
    }

    setFieldState(field, isValid, errorMessage) {
        const inputWrapper = field.closest('.input-wrapper');
        const errorElement = field.closest('.form-group').querySelector('.form-error');
        
        if (isValid) {
            inputWrapper.classList.remove('error');
            inputWrapper.classList.add('valid');
            errorElement.textContent = '';
        } else {
            inputWrapper.classList.remove('valid');
            inputWrapper.classList.add('error');
            errorElement.textContent = errorMessage;
            
            // Add shake animation
            inputWrapper.style.animation = 'shake 0.5s ease-in-out';
            setTimeout(() => {
                inputWrapper.style.animation = '';
            }, 500);
        }
    }

    clearError(field) {
        const inputWrapper = field.closest('.input-wrapper');
        const errorElement = field.closest('.form-group').querySelector('.form-error');
        
        inputWrapper.classList.remove('error');
        errorElement.textContent = '';
    }

    async handleLogin(e) {
        e.preventDefault();
        
        const email = document.getElementById('loginEmail').value.trim();
        const password = document.getElementById('loginPassword').value;
        const rememberMe = document.getElementById('rememberMe').checked;
        
        // Validate fields
        const emailField = document.getElementById('loginEmail');
        const passwordField = document.getElementById('loginPassword');
        
        const emailValid = this.validateField(emailField);
        const passwordValid = this.validateField(passwordField);
        
        if (!emailValid || !passwordValid) {
            return;
        }
        
        // Show loading state
        const submitBtn = e.target.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = `
            <i class="fas fa-spinner fa-spin"></i>
            <span>Iniciando...</span>
        `;
        submitBtn.disabled = true;
        
        // Simulate API call delay
        await new Promise(resolve => setTimeout(resolve, 1500));
        
        // Check credentials
        const user = this.users.find(u => u.email === email && u.password === password);
        
        if (user) {
            this.currentUser = user;
            localStorage.setItem('currentUser', JSON.stringify(user));
            
            if (rememberMe) {
                localStorage.setItem('rememberLogin', 'true');
            }
            
            window.campoVerdeApp.showToast('¡Bienvenido de vuelta! Redirigiendo...', 'success');
            
            setTimeout(() => {
                window.campoVerdeApp.navigateToPage('index.html');
            }, 1500);
        } else {
            window.campoVerdeApp.showToast('Email o contraseña incorrectos', 'error');
            
            // Reset button
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
            
            // Add error animation to form
            const form = e.target;
            form.style.animation = 'shake 0.5s ease-in-out';
            setTimeout(() => {
                form.style.animation = '';
            }, 500);
        }
    }

    async handleRegister(e) {
        e.preventDefault();
        
        const formData = {
            name: document.getElementById('registerName').value.trim(),
            lastName: document.getElementById('registerLastName').value.trim(),
            email: document.getElementById('registerEmail').value.trim(),
            phone: document.getElementById('registerPhone').value.trim(),
            password: document.getElementById('registerPassword').value,
            confirmPassword: document.getElementById('registerConfirmPassword').value
        };
        
        // Validate all fields
        let isValid = true;
        const fields = [
            'registerName', 'registerLastName', 'registerEmail', 
            'registerPhone', 'registerPassword', 'registerConfirmPassword'
        ];
        
        fields.forEach(fieldId => {
            const field = document.getElementById(fieldId);
            if (!this.validateField(field)) {
                isValid = false;
            }
        });
        
        // Check terms acceptance
        const acceptTerms = document.getElementById('acceptTerms');
        if (!acceptTerms.checked) {
            isValid = false;
            window.campoVerdeApp.showToast('Debes aceptar los términos y condiciones', 'warning');
        }
        
        if (!isValid) {
            return;
        }
        
        // Show loading state
        const submitBtn = e.target.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = `
            <i class="fas fa-spinner fa-spin"></i>
            <span>Creando cuenta...</span>
        `;
        submitBtn.disabled = true;
        
        // Simulate API call delay
        await new Promise(resolve => setTimeout(resolve, 2000));
        
        // Create new user
        const newUser = {
            id: this.generateId(),
            name: formData.name,
            lastName: formData.lastName,
            email: formData.email,
            phone: formData.phone,
            password: formData.password,
            createdAt: new Date().toISOString(),
            isAffiliate: false,
            membershipType: null
        };
        
        this.users.push(newUser);
        localStorage.setItem('users', JSON.stringify(this.users));
        
        window.campoVerdeApp.showToast('¡Cuenta creada exitosamente! Redirigiendo al login...', 'success');
        
        // Reset form
        e.target.reset();
        
        // Switch to login form
        setTimeout(() => {
            document.querySelector('.switch-btn[data-form="login"]').click();
            
            // Pre-fill email
            document.getElementById('loginEmail').value = formData.email;
            document.getElementById('loginEmail').parentElement.classList.add('focused');
        }, 1500);
        
        // Reset button
        setTimeout(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }, 1500);
    }

    handleSocialAuth(provider) {
        window.campoVerdeApp.showToast(`Autenticación con ${provider} no disponible en la demo`, 'info');
    }

    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substr(2);
    }
}

// Initialize auth system when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.authSystem = new AuthSystem();
});