// Advanced Animations for TheL4FPRO
class AnimationSystem {
    constructor() {
        this.init();
    }

    init() {
        this.setupIntersectionObserver();
        this.setupParallaxEffects();
        this.setupHoverEffects();
        this.setupScrollIndicators();
        this.initializeAnimations();
    }

    setupIntersectionObserver() {
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    this.triggerAnimation(entry.target);
                }
            });
        }, observerOptions);

        // Observe all elements with animation classes
        const animatedElements = document.querySelectorAll(
            '.feature-card, .benefit-card, .pricing-card, .faq-item, .stat-card, .hero-text > *, .animate-on-scroll'
        );
        
        animatedElements.forEach(element => {
            observer.observe(element);
        });
    }

    triggerAnimation(element) {
        if (element.classList.contains('feature-card')) {
            this.animateFeatureCard(element);
        } else if (element.classList.contains('benefit-card')) {
            this.animateBenefitCard(element);
        } else if (element.classList.contains('pricing-card')) {
            this.animatePricingCard(element);
        } else if (element.classList.contains('stat-card')) {
            this.animateStatCard(element);
        } else {
            this.animateGeneric(element);
        }
    }

    animateFeatureCard(card) {
        const delay = Array.from(card.parentElement.children).indexOf(card) * 100;
        
        setTimeout(() => {
            card.style.opacity = '1';
            card.style.transform = 'translateY(0) scale(1)';
            
            // Animate icon
            const icon = card.querySelector('.feature-icon i');
            if (icon) {
                setTimeout(() => {
                    icon.style.transform = 'scale(1.1)';
                    setTimeout(() => {
                        icon.style.transform = 'scale(1)';
                    }, 200);
                }, 300);
            }
        }, delay);
    }

    animateBenefitCard(card) {
        const delay = Array.from(card.parentElement.children).indexOf(card) * 150;
        
        setTimeout(() => {
            card.style.opacity = '1';
            card.style.transform = 'translateY(0)';
            
            // Animate benefit icon with bounce
            const iconContainer = card.querySelector('.benefit-icon');
            if (iconContainer) {
                setTimeout(() => {
                    iconContainer.style.animation = 'bounce 0.6s ease-out';
                }, 400);
            }
        }, delay);
    }

    animatePricingCard(card) {
        const delay = Array.from(card.parentElement.children).indexOf(card) * 200;
        
        setTimeout(() => {
            card.style.opacity = '1';
            card.style.transform = 'translateY(0) scale(1)';
            
            // Special animation for featured card
            if (card.classList.contains('featured')) {
                setTimeout(() => {
                    card.style.boxShadow = '0 20px 40px rgba(0, 0, 0, 0.15)';
                }, 500);
            }
        }, delay);
    }

    animateStatCard(card) {
        const delay = Array.from(card.parentElement.children).indexOf(card) * 100;
        
        setTimeout(() => {
            card.style.opacity = '1';
            card.style.transform = 'translateY(0)';
            
            // Animate number counting
            const numberElement = card.querySelector('.stat-number');
            if (numberElement) {
                this.animateCounter(numberElement);
            }
        }, delay);
    }

    animateGeneric(element) {
        element.style.opacity = '1';
        element.style.transform = 'translateY(0)';
    }

    animateCounter(element) {
        const target = element.textContent;
        const hasPlus = target.includes('+');
        const hasPercent = target.includes('%');
        const number = parseInt(target.replace(/[^\d]/g, '')) || 0;
        
        let current = 0;
        const increment = Math.max(1, Math.ceil(number / 60));
        const duration = 2000;
        const stepTime = duration / (number / increment);
        
        const timer = setInterval(() => {
            current += increment;
            if (current >= number) {
                current = number;
                clearInterval(timer);
            }
            
            let displayValue = current.toLocaleString();
            if (hasPlus) displayValue += '+';
            if (hasPercent) displayValue += '%';
            
            element.textContent = displayValue;
        }, stepTime);
    }

    setupParallaxEffects() {
        const parallaxElements = document.querySelectorAll('.hero-background, .floating-elements');
        
        window.addEventListener('scroll', () => {
            const scrolled = window.pageYOffset;
            const rate = scrolled * -0.5;
            
            parallaxElements.forEach(element => {
                element.style.transform = `translateY(${rate}px)`;
            });
        });
    }

    setupHoverEffects() {
        // Enhanced hover effects for cards
        const cards = document.querySelectorAll('.feature-card, .benefit-card, .pricing-card');
        
        cards.forEach(card => {
            card.addEventListener('mouseenter', () => {
                card.style.transform = 'translateY(-10px) scale(1.02)';
                card.style.boxShadow = '0 20px 40px rgba(0, 0, 0, 0.15)';
            });
            
            card.addEventListener('mouseleave', () => {
                card.style.transform = 'translateY(0) scale(1)';
                card.style.boxShadow = '';
            });
        });

        // Button hover effects
        const buttons = document.querySelectorAll('.btn');
        
        buttons.forEach(btn => {
            btn.addEventListener('mouseenter', () => {
                const glow = btn.querySelector('.btn-glow');
                if (glow) {
                    glow.style.opacity = '1';
                    glow.style.transform = 'scale(1)';
                }
            });
            
            btn.addEventListener('mouseleave', () => {
                const glow = btn.querySelector('.btn-glow');
                if (glow) {
                    glow.style.opacity = '0';
                    glow.style.transform = 'scale(0.8)';
                }
            });
        });
    }

    setupScrollIndicators() {
        // Create scroll progress indicator
        const progressBar = document.createElement('div');
        progressBar.className = 'scroll-progress';
        progressBar.innerHTML = '<div class="progress-fill"></div>';
        document.body.appendChild(progressBar);
        
        const progressFill = progressBar.querySelector('.progress-fill');
        
        window.addEventListener('scroll', () => {
            const scrollHeight = document.documentElement.scrollHeight - window.innerHeight;
            const scrolled = (window.pageYOffset / scrollHeight) * 100;
            progressFill.style.width = `${scrolled}%`;
        });
    }

    initializeAnimations() {
        // Initialize CSS custom properties for animations
        document.documentElement.style.setProperty('--animation-duration', '0.6s');
        document.documentElement.style.setProperty('--animation-easing', 'cubic-bezier(0.4, 0, 0.2, 1)');
        
        // Setup initial states for animated elements
        const animatedElements = document.querySelectorAll(
            '.feature-card, .benefit-card, .pricing-card, .stat-card'
        );
        
        animatedElements.forEach((element, index) => {
            element.style.opacity = '0';
            element.style.transform = 'translateY(50px) scale(0.95)';
            element.style.transition = `all var(--animation-duration) var(--animation-easing) ${index * 0.1}s`;
        });

        // Floating animations for background elements
        this.setupFloatingAnimations();
    }

    setupFloatingAnimations() {
        const floatingElements = document.querySelectorAll('.floating-ball, .floating-shape');
        
        floatingElements.forEach((element, index) => {
            const duration = 3 + Math.random() * 2;
            const delay = Math.random() * 2;
            const amplitude = 20 + Math.random() * 20;
            
            element.style.animation = `float ${duration}s ease-in-out ${delay}s infinite`;
            element.style.setProperty('--float-amplitude', `${amplitude}px`);
        });
    }

    // Utility method to create ripple effect
    createRipple(event) {
        const button = event.currentTarget;
        const rect = button.getBoundingClientRect();
        const size = Math.max(rect.width, rect.height);
        const x = event.clientX - rect.left - size / 2;
        const y = event.clientY - rect.top - size / 2;
        
        const ripple = document.createElement('span');
        ripple.className = 'ripple';
        ripple.style.width = ripple.style.height = size + 'px';
        ripple.style.left = x + 'px';
        ripple.style.top = y + 'px';
        
        button.appendChild(ripple);
        
        setTimeout(() => {
            ripple.remove();
        }, 600);
    }
}

// Initialize animation system
document.addEventListener('DOMContentLoaded', () => {
    window.animationSystem = new AnimationSystem();
    
    // Add ripple effect to buttons
    document.querySelectorAll('.btn').forEach(btn => {
        btn.addEventListener('click', window.animationSystem.createRipple);
    });
});

// CSS Animation Keyframes (injected dynamically)
const animationStyles = `
    @keyframes float {
        0%, 100% { transform: translateY(0px) rotate(0deg); }
        50% { transform: translateY(var(--float-amplitude, -20px)) rotate(180deg); }
    }
    
    @keyframes bounce {
        0%, 20%, 53%, 80%, 100% { transform: translate3d(0, 0, 0); }
        40%, 43% { transform: translate3d(0, -10px, 0); }
        70% { transform: translate3d(0, -5px, 0); }
        90% { transform: translate3d(0, -2px, 0); }
    }
    
    @keyframes pulse {
        0% { transform: scale(1); opacity: 1; }
        50% { transform: scale(1.05); opacity: 0.8; }
        100% { transform: scale(1); opacity: 1; }
    }
    
    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        10%, 30%, 50%, 70%, 90% { transform: translateX(-5px); }
        20%, 40%, 60%, 80% { transform: translateX(5px); }
    }
    
    .ripple {
        position: absolute;
        border-radius: 50%;
        background: rgba(255, 255, 255, 0.3);
        pointer-events: none;
        animation: ripple-animation 0.6s ease-out;
    }
    
    @keyframes ripple-animation {
        0% { transform: scale(0); opacity: 1; }
        100% { transform: scale(1); opacity: 0; }
    }
    
    .scroll-progress {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 3px;
        background: rgba(255, 255, 255, 0.1);
        z-index: 9999;
    }
    
    .progress-fill {
        height: 100%;
        background: linear-gradient(90deg, #10B981, #059669);
        transition: width 0.3s ease;
    }
`;

// Inject animation styles
const styleSheet = document.createElement('style');
styleSheet.textContent = animationStyles;
document.head.appendChild(styleSheet);