// Affiliates System JavaScript for TheL4FPRO
class AffiliatesSystem {
    constructor() {
        this.currentUser = JSON.parse(localStorage.getItem('currentUser') || 'null');
        this.isYearlyPricing = false;
        
        this.init();
    }

    init() {
        this.setupPricingToggle();
        this.setupPlanButtons();
        this.setupFAQ();
        this.setupScrollAnimations();
    }

    setupPricingToggle() {
        const toggle = document.getElementById('pricingToggle');
        if (toggle) {
            toggle.addEventListener('change', () => {
                this.isYearlyPricing = toggle.checked;
                this.updatePricing();
            });
        }
    }

    updatePricing() {
        const monthlyElements = document.querySelectorAll('.monthly');
        const yearlyElements = document.querySelectorAll('.yearly');
        
        if (this.isYearlyPricing) {
            monthlyElements.forEach(el => el.classList.add('hidden'));
            yearlyElements.forEach(el => el.classList.remove('hidden'));
        } else {
            monthlyElements.forEach(el => el.classList.remove('hidden'));
            yearlyElements.forEach(el => el.classList.add('hidden'));
        }

        // Add animation to pricing cards
        const pricingCards = document.querySelectorAll('.pricing-card');
        pricingCards.forEach((card, index) => {
            card.style.transform = 'scale(0.95)';
            setTimeout(() => {
                card.style.transform = 'scale(1)';
            }, index * 100);
        });
    }

    setupPlanButtons() {
        const planButtons = document.querySelectorAll('.plan-btn');
        
        planButtons.forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                
                const planCard = btn.closest('.pricing-card');
                const planName = planCard.querySelector('h3').textContent;
                
                this.handlePlanSelection(planCard, planName);
            });
        });
    }

    async handlePlanSelection(planCard, planName) {
        if (!this.currentUser) {
            window.campoVerdeApp.showToast('Debes iniciar sesión para suscribirte a un plan', 'warning');
            setTimeout(() => {
                window.campoVerdeApp.navigateToPage('login.html');
            }, 2000);
            return;
        }

        // Get plan details
        const priceElement = planCard.querySelector('.amount:not(.hidden)');
        const price = priceElement ? priceElement.textContent : '0';
        const period = this.isYearlyPricing ? 'anual' : 'mensual';

        // Show loading state
        const btn = planCard.querySelector('.plan-btn');
        const originalText = btn.innerHTML;
        btn.innerHTML = `
            <i class="fas fa-spinner fa-spin"></i>
            <span>Procesando...</span>
        `;
        btn.disabled = true;

        // Simulate subscription process
        await new Promise(resolve => setTimeout(resolve, 2000));

        // Update user membership
        this.currentUser.membershipType = planName.toLowerCase();
        this.currentUser.membershipPeriod = period;
        this.currentUser.membershipPrice = price;
        this.currentUser.membershipDate = new Date().toISOString();
        this.currentUser.isAffiliate = true;

        // Save to localStorage
        localStorage.setItem('currentUser', JSON.stringify(this.currentUser));

        // Update users array
        const users = JSON.parse(localStorage.getItem('users') || '[]');
        const userIndex = users.findIndex(u => u.id === this.currentUser.id);
        if (userIndex !== -1) {
            users[userIndex] = this.currentUser;
            localStorage.setItem('users', JSON.stringify(users));
        }

        // Show success message
        window.campoVerdeApp.showToast(`¡Bienvenido al plan ${planName}! Ya eres un afiliado premium.`, 'success');

        // Add visual feedback
        planCard.classList.add('selected');
        
        // Update button
        btn.innerHTML = `
            <i class="fas fa-check"></i>
            <span>Plan Activo</span>
        `;
        btn.classList.remove('btn-outline', 'btn-primary');
        btn.classList.add('btn-success');

        // Disable other plan buttons
        const otherButtons = document.querySelectorAll('.plan-btn');
        otherButtons.forEach(otherBtn => {
            if (otherBtn !== btn) {
                otherBtn.disabled = true;
                otherBtn.style.opacity = '0.5';
            }
        });

        // Show membership benefits
        setTimeout(() => {
            this.showMembershipBenefits(planName);
        }, 2000);
    }

    showMembershipBenefits(planName) {
        const benefits = this.getMembershipBenefits(planName);
        
        // Create benefits modal
        const modal = document.createElement('div');
        modal.className = 'modal-overlay benefits-modal';
        modal.innerHTML = `
            <div class="modal">
                <div class="modal-content">
                    <div class="modal-icon success">
                        <i class="fas fa-crown"></i>
                    </div>
                    <h3>¡Bienvenido al Plan ${planName}!</h3>
                    <p>Ya tienes acceso a todos estos beneficios exclusivos:</p>
                    <div class="benefits-list">
                        ${benefits.map(benefit => `
                            <div class="benefit-item animate-in">
                                <i class="fas fa-check-circle"></i>
                                <span>${benefit}</span>
                            </div>
                        `).join('')}
                    </div>
                    <div class="modal-actions">
                        <button class="btn btn-primary close-benefits">
                            <i class="fas fa-arrow-right"></i>
                            Comenzar a Disfrutar
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Show modal with animation
        setTimeout(() => {
            modal.classList.add('active');
        }, 100);
        
        // Setup close handler
        const closeBtn = modal.querySelector('.close-benefits');
        closeBtn.addEventListener('click', () => {
            modal.classList.remove('active');
            setTimeout(() => {
                document.body.removeChild(modal);
            }, 300);
        });
        
        // Animate benefits list
        const benefitItems = modal.querySelectorAll('.benefit-item');
        benefitItems.forEach((item, index) => {
            setTimeout(() => {
                item.style.opacity = '1';
                item.style.transform = 'translateX(0)';
            }, 500 + (index * 200));
        });
    }

    getMembershipBenefits(planName) {
        const benefits = {
            básico: [
                '15% descuento en todas las reservas',
                'Acceso completo al gimnasio',
                'Participación en torneos mensuales',
                'Soporte prioritario',
                'Aplicación móvil premium'
            ],
            premium: [
                '25% descuento en todas las reservas',
                'Acceso prioritario con 72h de anticipación',
                'Torneos exclusivos premium',
                'Equipo gratuito ocasional',
                'Entrenador personal mensual',
                'Acceso a eventos VIP'
            ],
            elite: [
                '35% descuento en todas las reservas',
                'Reservas ilimitadas',
                'Entrenamiento personalizado',
                'Acceso VIP a todos los eventos',
                'Área de descanso exclusiva',
                'Concierge personal'
            ]
        };
        
        return benefits[planName.toLowerCase()] || [];
    }

    setupFAQ() {
        const faqItems = document.querySelectorAll('.faq-item');
        
        faqItems.forEach(item => {
            const question = item.querySelector('.faq-question');
            const answer = item.querySelector('.faq-answer');
            const icon = question.querySelector('i');
            
            question.addEventListener('click', () => {
                const isOpen = item.classList.contains('active');
                
                // Close all other FAQs
                faqItems.forEach(otherItem => {
                    if (otherItem !== item) {
                        otherItem.classList.remove('active');
                        otherItem.querySelector('i').className = 'fas fa-plus';
                    }
                });
                
                // Toggle current FAQ
                if (isOpen) {
                    item.classList.remove('active');
                    icon.className = 'fas fa-plus';
                } else {
                    item.classList.add('active');
                    icon.className = 'fas fa-minus';
                }
            });
        });
    }

    setupScrollAnimations() {
        // Simple intersection observer for scroll animations
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.style.opacity = '1';
                    entry.target.style.transform = 'translateY(0)';
                }
            });
        }, observerOptions);

        // Observe elements with data-aos attributes
        const animatedElements = document.querySelectorAll('[data-aos]');
        animatedElements.forEach((element, index) => {
            element.style.opacity = '0';
            element.style.transform = 'translateY(50px)';
            element.style.transition = `all 0.8s cubic-bezier(0.4, 0, 0.2, 1) ${index * 0.1}s`;
            observer.observe(element);
        });

        // Special animations for hero stats
        const statCards = document.querySelectorAll('.stat-card');
        statCards.forEach((card, index) => {
            card.style.opacity = '0';
            card.style.transform = 'translateY(30px)';
            
            setTimeout(() => {
                card.style.transition = 'all 0.6s cubic-bezier(0.4, 0, 0.2, 1)';
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
                
                // Animate number counting
                this.animateNumber(card.querySelector('.stat-number'));
            }, 1000 + (index * 200));
        });
    }

    animateNumber(element) {
        const target = element.textContent;
        const isPercentage = target.includes('%');
        const number = parseInt(target.replace(/[^\d]/g, ''));
        
        let current = 0;
        const increment = Math.ceil(number / 50);
        
        const timer = setInterval(() => {
            current += increment;
            if (current >= number) {
                current = number;
                clearInterval(timer);
            }
            
            element.textContent = isPercentage ? `${current}%` : 
                                 target.includes('+') ? `${current.toLocaleString()}+` : 
                                 target.includes('/') ? `${current}/7` : 
                                 current.toLocaleString();
        }, 30);
    }
}

// Initialize affiliates system when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    if (document.body.classList.contains('affiliates-page')) {
        window.affiliatesSystem = new AffiliatesSystem();
    }
});