class AcademyRegistration {
    constructor() {
        this.currentStep = 0;
        this.totalSteps = document.querySelectorAll('.form-step').length;
        this.formData = {};
        this.init();
    }

    init() {
        this.setupStepNavigation();
        this.setupFileUpload();
        this.setupConditionalFields();
        this.setupFormValidation();
        this.cacheRequiredStates();
        this.applyRequiredForStep(this.currentStep);
        this.updateProgress();
    }

    setupStepNavigation() {
        const nextBtn = document.getElementById('nextBtn');
        const prevBtn = document.getElementById('prevBtn');
        const form = document.getElementById('academyForm');
        if (nextBtn) nextBtn.addEventListener('click', () => this.nextStep());
        if (prevBtn) prevBtn.addEventListener('click', () => this.prevStep());
        if (form) form.addEventListener('submit', (e) => this.handleSubmit(e));
    }

    setupFileUpload() {
        const fileUploadArea = document.getElementById('fileUploadArea');
        const fileInput = document.getElementById('playerPhoto');
        const uploadPreview = document.getElementById('uploadPreview');
        const uploadPlaceholder = fileUploadArea?.querySelector('.upload-placeholder');
        const removeBtn = document.getElementById('removeFile');
        if (fileUploadArea && fileInput) {
            fileUploadArea.addEventListener('click', () => {
                fileInput.click();
            });
            fileUploadArea.addEventListener('dragover', (e) => {
                e.preventDefault();
                fileUploadArea.style.borderColor = 'var(--brand)';
                fileUploadArea.style.background = 'var(--brand-pale)';
            });
            fileUploadArea.addEventListener('dragleave', () => {
                fileUploadArea.style.borderColor = '';
                fileUploadArea.style.background = '';
            });
            fileUploadArea.addEventListener('drop', (e) => {
                e.preventDefault();
                fileUploadArea.style.borderColor = '';
                fileUploadArea.style.background = '';
                const files = e.dataTransfer.files;
                if (files.length > 0) {
                    this.handleFileSelect(files[0]);
                }
            });
            fileInput.addEventListener('change', (e) => {
                if (e.target.files.length > 0) {
                    this.handleFileSelect(e.target.files[0]);
                }
            });
            if (removeBtn) {
                removeBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.removeFile();
                });
            }
        }
    }

    handleFileSelect(file) {
        const fileInput = document.getElementById('playerPhoto');
        const uploadPreview = document.getElementById('uploadPreview');
        const uploadPlaceholder = document.querySelector('.upload-placeholder');
        const previewImage = document.getElementById('previewImage');
        const fileName = document.getElementById('fileName');
        if (!this.validateFile(file)) return;
        const dt = new DataTransfer();
        dt.items.add(file);
        fileInput.files = dt.files;
        const reader = new FileReader();
        reader.onload = (e) => {
            if (previewImage) previewImage.src = e.target.result;
            if (fileName) fileName.textContent = file.name;
            if (uploadPreview) uploadPreview.style.display = 'flex';
            if (uploadPlaceholder) uploadPlaceholder.style.display = 'none';
        };
        reader.readAsDataURL(file);
    }

    validateFile(file) {
        if (!file.type.startsWith('image/')) {
            this.showToast('Por favor, selecciona un archivo de imagen válido.', 'error');
            return false;
        }
        if (file.size > 10 * 1024 * 1024) {
            this.showToast('El archivo es demasiado grande. Máximo 10MB.', 'error');
            return false;
        }
        return true;
    }

    removeFile() {
        const fileInput = document.getElementById('playerPhoto');
        const uploadPreview = document.getElementById('uploadPreview');
        const uploadPlaceholder = document.querySelector('.upload-placeholder');
        if (fileInput) fileInput.value = '';
        if (uploadPreview) uploadPreview.style.display = 'none';
        if (uploadPlaceholder) uploadPlaceholder.style.display = 'block';
    }

    setupConditionalFields() {
        const conditionalFields = document.querySelectorAll('.conditional-field');
        conditionalFields.forEach(field => {
            const condition = field.getAttribute('data-show-if');
            if (condition) {
                const [fieldName, value] = condition.split('=');
                const triggerInputs = document.querySelectorAll(`input[name="${fieldName}"]`);
                triggerInputs.forEach(input => {
                    input.addEventListener('change', () => {
                        const shouldShow = input.value === value && input.checked;
                        const allFieldsWithCondition = document.querySelectorAll(`[data-show-if="${condition}"]`);
                        allFieldsWithCondition.forEach(condField => {
                            condField.style.display = shouldShow ? 'block' : 'none';
                            const requiredInputs = condField.querySelectorAll('input[required], textarea[required]');
                            requiredInputs.forEach(req => {
                                req.required = shouldShow && (req.dataset.requiredOriginal === '1');
                            });
                        });
                    });
                });
            }
        });
    }

    setupFormValidation() {
        const form = document.getElementById('academyForm');
        const inputs = form.querySelectorAll('input, textarea, select');
        inputs.forEach(input => {
            input.addEventListener('blur', () => this.validateField(input));
            input.addEventListener('input', () => this.clearFieldError(input));
        });
    }

    validateField(field) {
        if (field.id === 'firstName') {
            const v = field.value.trim();
            if (v.length < 2) {
                this.setFieldError(field, 'Introduce tu nombre.');
                return false;
            }
        }
        if (field.id === 'lastName') {
            const v = field.value.trim();
            if (v.length < 2) {
                this.setFieldError(field, 'Introduce tus apellidos.');
                return false;
            }
        }
        if (field.required && !field.value.trim()) {
            this.setFieldError(field, 'Este campo es obligatorio');
            return false;
        }
        if (field.type === 'email' && field.value) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(field.value)) {
                this.setFieldError(field, 'Por favor, introduce un email válido');
                return false;
            }
        }
        this.clearFieldError(field);
        return true;
    }

    setFieldError(field, message) {
        const wrapper = field.closest('.input-wrapper') || field.closest('.form-group');
        if (!wrapper) return;
        wrapper.classList.add('error');
        let errorElement = wrapper.querySelector('.field-error');
        if (!errorElement) {
            errorElement = document.createElement('div');
            errorElement.className = 'field-error';
            wrapper.appendChild(errorElement);
        }
        errorElement.textContent = message;
    }

    clearFieldError(field) {
        const wrapper = field.closest('.input-wrapper') || field.closest('.form-group');
        if (!wrapper) return;
        wrapper.classList.remove('error');
        const errorElement = wrapper.querySelector('.field-error');
        if (errorElement) errorElement.remove();
    }

    validateCurrentStep() {
        const currentStepElement = document.querySelector(`.form-step[data-step="${this.currentStep}"]`);
        if (!currentStepElement) return true;
        const requiredFields = currentStepElement.querySelectorAll('input[required], textarea[required], select[required]');
        let isValid = true;
        requiredFields.forEach(field => {
            if (!this.validateField(field)) isValid = false;
        });
        if (this.currentStep === 1) {
            const first = document.getElementById('firstName');
            const last = document.getElementById('lastName');
            if (first && !this.validateField(first)) isValid = false;
            if (last && !this.validateField(last)) isValid = false;
        }
        return isValid;
    }

    nextStep() {
        if (!this.validateCurrentStep()) {
            this.showToast('Por favor, completa todos los campos obligatorios.', 'warning');
            return;
        }
        if (this.currentStep < this.totalSteps - 1) {
            this.currentStep++;
            this.showStep(this.currentStep);
        }
    }

    prevStep() {
        if (this.currentStep > 0) {
            this.currentStep--;
            this.showStep(this.currentStep);
        }
    }

    showStep(stepNumber) {
        document.querySelectorAll('.form-step').forEach(step => {
            step.classList.remove('active');
        });
        const currentStep = document.querySelector(`.form-step[data-step="${stepNumber}"]`);
        if (currentStep) currentStep.classList.add('active');
        this.applyRequiredForStep(stepNumber);
        this.updateProgress();
        this.updateStepIndicators();
        this.updateNavigationButtons();
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }

    updateProgress() {
        const progressFill = document.getElementById('progressFill');
        const progress = ((this.currentStep + 1) / this.totalSteps) * 100;
        if (progressFill) progressFill.style.width = `${progress}%`;
    }

    updateStepIndicators() {
        const stepIndicators = document.querySelectorAll('.step');
        stepIndicators.forEach((step, index) => {
            step.classList.remove('active', 'completed');
            if (index === this.currentStep) step.classList.add('active');
            else if (index < this.currentStep) step.classList.add('completed');
        });
    }

    updateNavigationButtons() {
        const nextBtn = document.getElementById('nextBtn');
        const prevBtn = document.getElementById('prevBtn');
        const submitBtn = document.getElementById('submitBtn');
        if (prevBtn) prevBtn.style.display = this.currentStep > 0 ? 'flex' : 'none';
        if (this.currentStep === this.totalSteps - 1) {
            if (nextBtn) nextBtn.style.display = 'none';
            if (submitBtn) submitBtn.style.display = 'flex';
        } else {
            if (nextBtn) nextBtn.style.display = 'flex';
            if (submitBtn) submitBtn.style.display = 'none';
        }
    }

    async handleSubmit(e) {
        e.preventDefault();
        this.ensureCurrentStepRequired();
        if (!this.validateCurrentStep()) {
            this.showToast('Por favor, completa todos los campos obligatorios.', 'warning');
            return;
        }
        const first = document.getElementById('firstName');
        const last = document.getElementById('lastName');
        let ok = true;
        if (first && !this.validateField(first)) ok = false;
        if (last && !this.validateField(last)) ok = false;
        if (!ok) {
            this.showToast('Por favor, indica nombre y apellidos.', 'warning');
            return;
        }
        const form = document.getElementById('academyForm');
        const submitBtn = document.getElementById('submitBtn');
        const originalHTML = submitBtn ? submitBtn.innerHTML : '';
        if (submitBtn) {
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Enviando...</span>';
            submitBtn.disabled = true;
        }
        try {
            const fd = new FormData(form);
            const res = await fetch('/academy-register', {
                method: 'POST',
                body: fd
            });
            let out = {};
            try {
                out = await res.json();
            } catch {
                throw new Error('Respuesta inesperada del servidor.');
            }
            if (!res.ok || !out.ok) {
                const msg = out?.detail || out?.error || 'Error al guardar el formulario.';
                throw new Error(msg);
            }
            window.location.href = out.redirect || '/login';
            return;
        } catch (error) {
            this.showToast(error?.message || 'Error al enviar el formulario. Inténtalo de nuevo.', 'error');
            if (submitBtn) {
                submitBtn.innerHTML = originalHTML;
                submitBtn.disabled = false;
            }
        }
    }

    showSuccessModal(pin, redirectUrl) {
        const modal = document.getElementById('successModal');
        const pinEl = document.getElementById('pinCode');
        const continueBtn = document.getElementById('continueBtn');
        if (pinEl && typeof pin === 'string') pinEl.textContent = pin;
        if (modal) {
            modal.classList.add('active');
            if (continueBtn) {
                continueBtn.onclick = () => {
                    window.location.href = redirectUrl || '/complete-registration';
                };
            } else {
                setTimeout(() => {
                    window.location.href = redirectUrl || '/complete-registration';
                }, 1200);
            }
        } else {
            alert('Tu PIN temporal es: ' + pin + '\nGuárdalo: caduca en unas horas.');
            window.location.href = redirectUrl || '/complete-registration';
        }
    }

    showToast(message, type = 'info') {
        const toast = document.createElement('div');
        toast.style.cssText = `
            position: fixed;
            top: 2rem;
            right: 2rem;
            background: ${type === 'success' ? '#10b981' : type === 'warning' ? '#f59e0b' : type === 'error' ? '#ef4444' : '#10b981'};
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 10px;
            z-index: 9999;
            box-shadow: 0 10px 25px rgba(0,0,0,0.3);
            transform: translateX(100%);
            transition: transform 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            backdrop-filter: blur(10px);
        `;
        const iconClass = type === 'success' ? 'fa-check-circle' : type === 'warning' ? 'fa-exclamation-triangle' : type === 'error' ? 'fa-exclamation-circle' : 'fa-info-circle';
        toast.innerHTML = `<i class="fas ${iconClass}"></i><span>${message}</span>`;
        document.body.appendChild(toast);
        setTimeout(() => {
            toast.style.transform = 'translateX(0)';
        }, 100);
        setTimeout(() => {
            toast.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (document.body.contains(toast)) {
                    document.body.removeChild(toast);
                }
            }, 300);
        }, 4000);
    }

    cacheRequiredStates() {
        const allControls = document.querySelectorAll('.form-step input, .form-step textarea, .form-step select');
        allControls.forEach(el => {
            if (el.dataset.requiredOriginal == null) {
                el.dataset.requiredOriginal = el.required ? '1' : '0';
            }
        });
    }

    applyRequiredForStep(stepNumber) {
        const steps = document.querySelectorAll('.form-step');
        steps.forEach(step => {
            const isActive = step.getAttribute('data-step') == String(stepNumber);
            const controls = step.querySelectorAll('input, textarea, select');
            controls.forEach(el => {
                const wasRequired = el.dataset.requiredOriginal === '1';
                let shouldRequire = wasRequired;
                const cond = el.closest('.conditional-field');
                if (cond && window.getComputedStyle(cond).display === 'none') {
                    shouldRequire = false;
                }
                el.required = isActive ? shouldRequire : false;
            });
        });
    }

    ensureCurrentStepRequired() {
        this.applyRequiredForStep(this.currentStep);
    }
}

document.addEventListener('DOMContentLoaded', () => {
    window.academyRegistration = new AcademyRegistration();
});

const errorStyles = `
    .input-wrapper.error input,
    .input-wrapper.error textarea,
    .input-wrapper.error select {
        border-color: #ef4444;
        background: rgba(239, 68, 68, 0.05);
        box-shadow: 0 0 0 3px rgba(239, 68, 68, 0.1);
    }
    .field-error {
        color: #ef4444;
        font-size: 0.875rem;
        margin-top: 0.5rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }
    .field-error::before {
        content: '⚠';
        font-size: 1rem;
    }
`;
const styleSheet = document.createElement('style');
styleSheet.textContent = errorStyles;
document.head.appendChild(styleSheet);
